/*
	DooM MaP StaTistics, by Frans P. de Vries.

Derived from:

	DooM PostScript Maps Utility, by Frans P. de Vries.

And thus from:

	Doom Editor Utility, by Brendon Wyber and Raphaël Quinet.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	STATS.C - Statistics routines.
*/

#include "dmmpst.h"
#include "stats.h"
#include "levels.h"
#include "things.h"


/* map attribute definitions */
#define L_TELEP1 39       /* one-time teleport linedef. all in Doom format */
#define L_TELEPR 97       /* repeatable teleport linedef */
#define L_TELPM1 125      /* one-time monster-only teleport linedef */
#define L_TELPMR 126      /* repeatable monster-only teleport linedef */
#define S_SECRET 9        /* secret sector */
#define B_GENSEC  0x0080  /* Boom: generalized secret sector, Doom format */
#define B_GENSECH 0x0400  /* Boom: generalized secret sector, Hexen format */


/* local function definitions */
TMPL_varlist *tmpl_add_names( TMPL_varlist *, char *);
TMPL_varlist *tmpl_add_slot( TMPL_varlist *, BCINT, BCINT);
TMPL_varlist *tmpl_add_number( TMPL_varlist *, BCINT);
TMPL_varlist *tmpl_add_stats( TMPL_varlist *);
TMPL_varlist *tmpl_add_things( TMPL_varlist *);

void AggregateSpecials( void);
Bool ThingInMode( UBCINT, int);

/* local variables */
Bool splitDM[ThClass_End];


/*
	read level & analyze its statistics
*/
void AnalyzeLevel( BCINT episode, BCINT mission)
{
	TMPL_varlist *tpldata = NULL;
	TMPL_loop    *tplloop = NULL;
	char *wktpfile = NULL;
	int *secrets = NULL;
	int n, m, gensec, secret = 0, telept = 0;
	BCINT lumpver;
	BCLNG k;

	/* read data of this level */
	lumpver = ReadLevelData( episode, mission);

	/* count secret Sectors & teleporting LineDefs */
	gensec = ((lumpver & 0xF0) == 0x40) ? B_GENSECH : B_GENSEC; // Hexen or Doom format
	for (n = 0; n < NumSectors; n++)
		if (Sectors[n].special == S_SECRET ||
		    (Sectors[n].special & gensec) == gensec)
			secret++;
	if ((GameVersion & 0xF0) != 0x40) // skip for Hexen format
		for (n = 0; n < NumLineDefs; n++)
			if (LineDefs[n].type == L_TELEP1 || LineDefs[n].type == L_TELEPR)
				telept++;

	/* collect list of secret Sectors */
	if (secret > 0)
	{
		secrets = (int *) GetMemory( secret * sizeof( int));
		m = 0;
		for (n = 0; n < NumSectors; n++)
			if (Sectors[n].special == S_SECRET)
			{
				secrets[m] = n;
				m++;
			}
			else if ((Sectors[n].special & gensec) == gensec)
			{
				secrets[m] = -n;
				m++;
			}
	}

	/* check and warn if Hexen-format flags exist but aren't counted */
	if (lumpver != GameVersion && !WBoomMP)
		printf( "Warning: level's Things contain Hexen-format flags.  Use option '+b'.\n");
	/* check and warn if Boom MP flags exist but aren't counted */
	else if (CheckThingBoomFlags() && !WBoomMP)
		printf( "Warning: level's Things contain Boom MP flags.  Use option '+b'.\n");

	/* initialize and analyze thing statistics */
	switch (GameVersion & 0xF0)
	{
		case 0x00: InitDoomThings(); break;
		case 0x10: InitHereticThings(); break;
		case 0x20: InitStrifeThings( mission); break;
		case 0x40: InitHexenThings(); break;
	}
	if (CustomMap != NULL)
		ReadCustomThings();
	AnalyzeThings( lumpver, TRUE);

	/* print verbose statistics */
	if (Verbose)
	{
		printf( "\n");
		if (GameVersion == 0x02 || GameVersion >= 0x20)
		{
			switch (GameVersion)
			{
				case 0x02: printf( "# DOOM II"); break;
				case 0x20: printf( "# Strife Demo"); break;
				case 0x22: printf( "# Strife Full"); break;
				case 0x40: printf( "# Hexen Demo"); break;
				case 0x42: printf( "# Hexen Full"); break;
				case 0x48: printf( "# Hexen: DotDC"); break;
				default: ProgError( "unsupported game version (%02x)", GameVersion);
			}
			printf( "  Map statistics for level MAP%02d: %s\n\n", mission,
					  ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
		}
		else // < 0x20
		{
			switch (GameVersion)
			{
				case 0x00: printf( "# Shareware DOOM"); break;
				case 0x01: printf( "# Registered DOOM"); break;
				case 0x04: printf( "# Ultimate DOOM"); break;
				case 0x10: printf( "# Shareware Heretic"); break;
				case 0x11: printf( "# Registered Heretic"); break;
				case 0x14: printf( "# Heretic: SotSR"); break;
				default: ProgError( "unsupported game version (%02x)", GameVersion);
			}
			printf( "  Map statistics for level E%dM%d: %s\n\n", episode, mission,
					  ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
		}

		printf( "MapDim:\t  MaxX\t  MinX\t SizeX\t  MaxY\t  MinY\t SizeY\n");
		printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
				  MapMaxX, MapMinX, MapMaxX - MapMinX, MapMaxY, MapMinY, MapMaxY - MapMinY);

		printf( "\n");
		printf( "Struct:\tThings\tVertxs\tBefore\tLinDfs\tTelePt\tSidDfs\tSectrs\tSecret\n");
		printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
				  NumThings, TotVertexes, NumVertexes, NumLineDefs, telept, NumSideDefs, NumSectors, secret);
		printf( "\n");

		/* print list of secret sectors */
		if (secret > 0)
		{
			printf( "Secrets:");
			for (n = 0; n < secret; n++)
			{
				if (secrets[n] >= 0)
					printf( "%6d\t", secrets[n]);
				else
					printf( "%4d:B\t", -secrets[n]);
				if (((n+1) % 8) == 0 && (n+1) < secret)
					printf( "\n\t");
			}
			printf( "\n\n");
		}
	}

	/* collect classes for split DM table */
	if (WSplitDM > 0)
	{
		for (n = 0; n < ThClass_End; n++)
			splitDM[n] = FALSE;
		for (k = WSplitDM; k > 0; k = k / 10) {
			splitDM[k % 10 - 1] = TRUE;
		}
		//for (n = 0; n < ThClass_End; n++)
		//	printf( "class %d: %s\n", n, (splitDM[n] ? "2nd" : "1st"));
	}

	/* output DoomWiki file */
	if (WkFile != NULL)
	{
		/* process aggregate values */
		AggregateSpecials();

		if (WSkeleton)
		{
			/* add map attributes */
			tpldata = tmpl_add_names( tpldata, (UserLvlNm != NULL ? UserLvlNm : LevelName));
			tpldata = tmpl_add_slot( tpldata, episode, mission);
			tpldata = tmpl_add_number( tpldata, mission);

			/* add secret sector values */
			if (secret > 0)
				for (n = 0; n < secret; n++)
					tplloop = TMPL_add_varlist( tplloop,
						tmpl_add_int( NULL, "SECRET", abs(secrets[n])));
			tpldata = TMPL_add_loop( tpldata, "SECRETS", tplloop);

			/* add map_data.tpl & things.tpl values */
			tpldata = tmpl_add_stats( tpldata);
			tpldata = tmpl_add_things( tpldata);

			/* define and fill DoomWiki template */
			if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "skeleton.tpl") == -1)
				ProgError( "unable to concatenate template path");
			TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
			TMPL_free_varlist( tpldata);
			tpldata = NULL;
			FreeMemory( wktpfile);
		}
		else if (WSecrets)
		{
			/* add secret sector values */
			if (secret > 0)
				for (n = 0; n < secret; n++)
					tplloop = TMPL_add_varlist( tplloop,
						tmpl_add_int( NULL, "SECRET", abs(secrets[n])));
			tpldata = TMPL_add_loop( tpldata, "SECRETS", tplloop);

			/* define and fill DoomWiki template */
			if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "secrets.tpl") == -1)
				ProgError( "unable to concatenate template path");
			TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
			TMPL_free_varlist( tpldata);
			tpldata = NULL;
			FreeMemory( wktpfile);
		}
		else
		{
			if (WStatistics)
				fprintf( WkFile, "== Statistics ==\n");
			if (WStatistics || WMapdata)
			{
				/* add map_data.tpl values */
				tpldata = tmpl_add_stats( tpldata);

				/* define and fill DoomWiki template */
				if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "map_data.tpl") == -1)
					ProgError( "unable to concatenate template path");
				TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
				TMPL_free_varlist( tpldata);
				tpldata = NULL;
				FreeMemory( wktpfile);
			}
			if (WStatistics)
				fprintf( WkFile, "\n");
			if (WStatistics || WThings)
			{
				/* add things.tpl values */
				tpldata = tmpl_add_things( tpldata);

				/* define and fill DoomWiki template */
				if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "things.tpl") == -1)
					ProgError( "unable to concatenate template path");
				TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
				TMPL_free_varlist( tpldata);
				tpldata = NULL;
				FreeMemory( wktpfile);
			}
		}
	}

	/* clean up & free space */
	if (secret > 0)
		FreeMemory( secrets);

	ForgetLevelData();
	ForgetThings();
}


/*
	add complete and sortable map names to template variables
*/
TMPL_varlist *tmpl_add_names( TMPL_varlist *tpl, char *name)
{
	TMPL_varlist *t = tpl;
	char *n = name;

	if (strlen(name) == 0)
		return t;
	t = TMPL_add_var( t, "MAPNAME", name, NULL);

	/* check if name starts with an article */
	if (strncmp( name, "A ", 2) == 0)
		n = &name[2];
	if (strncmp( name, "An ", 3) == 0)
		n = &name[3];
	if (strncmp( name, "The ", 4) == 0)
		n = &name[4];
	/* add sortable name without article */
	if (n != name)
		t = TMPL_add_var( t, "MAPSORT", n, NULL);

	return t;
}

/*
	add map slot & game to template variables
*/
TMPL_varlist *tmpl_add_slot( TMPL_varlist *tpl, BCINT epi, BCINT mis)
{
	TMPL_varlist *t = tpl;
	char strbuf[12];

	if (GameVersion == 0x02 || GameVersion >= 0x20)
	{
		snprintf( strbuf, sizeof(strbuf), "MAP%02d", mis);
		t = TMPL_add_var( t, "MAPSLOT", strbuf, NULL);
		if (UserGamNm != NULL)
		{
			t = TMPL_add_var( t, "MAPGAME", UserGamNm, NULL);
		}
		else
		{
			if (GameVersion == 0x02)
				t = TMPL_add_var( t, "MAPGAME", "Doom II", NULL);
			else if ((GameVersion & 0xF0) == 0x20)
				t = TMPL_add_var( t, "MAPGAME", "Strife", NULL);
			else // == 0x40
				t = TMPL_add_var( t, "MAPGAME", "Hexen", NULL);
		}
	}
	else // < 0x20
	{
		snprintf( strbuf, sizeof(strbuf), "E%dM%d", epi, mis);
		t = TMPL_add_var( t, "MAPSLOT", strbuf, NULL);
		if (UserGamNm != NULL)
		{
			t = TMPL_add_var( t, "MAPGAME", UserGamNm, NULL);
		}
		else
		{
			t = TMPL_add_var( t, "MAPGAME",
			                  ((GameVersion & 0x10) != 0x00 ? "Heretic": "Doom"), NULL);
		}
	}

	return t;
}

/*
	add map number'th in words to template variables
*/
TMPL_varlist *tmpl_add_number( TMPL_varlist *tpl, BCINT mis)
{
	char strbuf[24];

	memset( strbuf, 0, sizeof(strbuf));
	if (GameVersion == 0x02)
	{
		switch (mis)
		{
			case  1: sprintf( strbuf, "first"); break;
			case  2: sprintf( strbuf, "second"); break;
			case  3: sprintf( strbuf, "third"); break;
			case  4: sprintf( strbuf, "fourth"); break;
			case  5: sprintf( strbuf, "fifth"); break;
			case  6: sprintf( strbuf, "sixth"); break;
			case  7: sprintf( strbuf, "seventh"); break;
			case  8: sprintf( strbuf, "eighth"); break;
			case  9: sprintf( strbuf, "ninth"); break;
			case 10: sprintf( strbuf, "tenth"); break;
			case 11: sprintf( strbuf, "eleventh"); break;
			case 12: sprintf( strbuf, "twelfth"); break;
			case 13: sprintf( strbuf, "thirteenth"); break;
			case 14: sprintf( strbuf, "fourteenth"); break;
			case 15: sprintf( strbuf, "fifteenth"); break;
			case 16: sprintf( strbuf, "sixteenth"); break;
			case 17: sprintf( strbuf, "seventeenth"); break;
			case 18: sprintf( strbuf, "eighteenth"); break;
			case 19: sprintf( strbuf, "nineteenth"); break;
			case 20: sprintf( strbuf, "twentieth"); break;
			case 21: sprintf( strbuf, "twenty-first"); break;
			case 22: sprintf( strbuf, "twenty-second"); break;
			case 23: sprintf( strbuf, "twenty-third"); break;
			case 24: sprintf( strbuf, "twenty-fourth"); break;
			case 25: sprintf( strbuf, "twenty-fifth"); break;
			case 26: sprintf( strbuf, "twenty-sixth"); break;
			case 27: sprintf( strbuf, "twenty-seventh"); break;
			case 28: sprintf( strbuf, "twenty-eighth"); break;
			case 29: sprintf( strbuf, "twenty-ninth"); break;
			case 30: sprintf( strbuf, "thirtieth and final"); break;
			case 31: sprintf( strbuf, "first secret"); break;
			case 32: sprintf( strbuf, "second secret"); break;
			default: sprintf( strbuf, "%d-th", mis); break;
		}
	}
	else if (GameVersion >= 0x20)
		; // no number'th convention in Hexen and Strife
	else // < 0x20
	{
		switch (mis)
		{
			case 1: sprintf( strbuf, "first"); break;
			case 2: sprintf( strbuf, "second"); break;
			case 3: sprintf( strbuf, "third"); break;
			case 4: sprintf( strbuf, "fourth"); break;
			case 5: sprintf( strbuf, "fifth"); break;
			case 6: sprintf( strbuf, "sixth"); break;
			case 7: sprintf( strbuf, "seventh"); break;
			case 8: sprintf( strbuf, "eighth and final"); break;
			case 9: sprintf( strbuf, "ninth and secret"); break;
		}
	}

	return TMPL_add_var( tpl, "MAPNUMTH", strbuf, NULL);
}

/*
	add map statistics to template variables
*/
TMPL_varlist *tmpl_add_stats( TMPL_varlist *tpl)
{
	TMPL_varlist *t = tpl;

	t = tmpl_add_int( t, "NUMTHNG", NumThings);
	t = tmpl_add_int( t, "NUMVERT", TotVertexes);
	t = tmpl_add_int( t, "NUMLINE", NumLineDefs);
	t = tmpl_add_int( t, "NUMSIDE", NumSideDefs);
	t = tmpl_add_int( t, "NUMSECT", NumSectors);
	t = tmpl_add_int( t, "NUMVBEF", NumVertexes);

	return t;
}

/*
	add map things to template variables
*/
TMPL_varlist *tmpl_add_things( TMPL_varlist *tpl)
{
	TMPL_loop    *tlmodes = NULL, *tlclasses, *tlitems;
	TMPL_varlist *tmode, *tclass, *titem;
	int m, c, i, class, modecount = 0;
	Bool secondDM = FALSE;

	/* add game-specific skills anchor */
	tpl = tmpl_add_skill( tpl);

	/* process all relevant modes */
	for (m = 0; m < ThMode_End; m++)
	{
		if ((GameVersion & 0xF0) == 0x40) // Hexen format
		{
			/* collect items for these modes:
			   1 = Cleric, 2 = Fighter, 3 = Mage, 5 = CooP, 6 = DM */
			if (m == ThModeSP || m == ThModeMP)
				continue;
		}
		else // Doom format
		{
			/* collect items for these modes:
			   0 = SP, 4 = MP (if !WBoomMP), 5 = CooP, 6 = DM (both if WBoomMP) */
			if (WBoomMP && m == ThModeMP)
				continue;
			if (!WBoomMP && (m == ThModeCP || m == ThModeDM))
				continue;
			if (m == ThModeXC || m == ThModeXF || m == ThModeXM)
				continue;
		}
		/* if excluding SP/Coop tables, collect only 4 = MP, 6 = DM */
		if (WXcludeSP &&
		    (m == ThModeSP || m == ThModeXC || m == ThModeXF || m == ThModeXM || m == ThModeCP))
			continue;

		/* process all thing classes */
		tlclasses = NULL;
		for (c = 0; c < NumThClasses; c++)
		{
			class = ThClasses[c]->class;
			/* special check in DM */
			if (m == ThModeDM)
			{
				/* skip Keys class */
				if (class == ThClassKeys)
					continue;
			}
			/* special check in MP-only/DM */
			if ((m == ThModeMP && WXcludeSP) || m == ThModeDM)
			{
				/* skip classes for 1st or 2nd DM column */
				if ((splitDM[class] && !secondDM) || (!splitDM[class] && secondDM))
					continue;
			}

			/* collect item(s) for this class */
			tlitems = NULL;
			for (i = 0; i < NumThItems; i++)
			{
				/* collect item if it exists and should be included in this mode */
				if (class == ThItems[i]->class && ThingInMode( ThItems[i]->type, m) &&
				    (ThItems[i]->easy[m] + ThItems[i]->medm[m] + ThItems[i]->hard[m]) > 0)
				{
					titem = TMPL_add_var( NULL, "THITEM",
					                      UsedWiki( ThItems[i]->wikiname,
					                                ThItems[i]->wikilink,
					                                ThItems[i]->wikistr), NULL);
					titem = tmpl_add_int( titem, "THEASY", ThItems[i]->easy[m]);
					titem = tmpl_add_int( titem, "THMEDM", ThItems[i]->medm[m]);
					titem = tmpl_add_int( titem, "THHARD", ThItems[i]->hard[m]);
					/* check whether to collate row into one value */
					if (ThItems[i]->easy[m] == ThItems[i]->medm[m] &&
					    ThItems[i]->easy[m] == ThItems[i]->hard[m])
						titem = TMPL_add_var( titem, "THEQUAL", "equal", NULL);
					tlitems = TMPL_add_varlist( tlitems, titem);
				}
			}
			/* add class section if it contains item(s) */
			if (tlitems != NULL)
			{
				tclass = TMPL_add_var( NULL, "THCLASS",
				                       UsedWiki( ThClasses[c]->wikiname,
				                                 ThClasses[c]->wikilink,
				                                 ThClasses[c]->wikistr), NULL);
				tclass = TMPL_add_loop( tclass, "THITEMS", tlitems);
				tlclasses = TMPL_add_varlist( tlclasses, tclass);
			}
		}
		/* add mode section if it contains class(es) */
		if (tlclasses != NULL)
		{
			switch (m)
			{
				case ThModeSP: tmode = TMPL_add_var( NULL, "THMODE", "Single-player", NULL); break;
				case ThModeXC: tmode = TMPL_add_var( NULL, "THMODE", "Cleric single-player", NULL); break;
				case ThModeXF: tmode = TMPL_add_var( NULL, "THMODE", "Fighter single-player", NULL); break;
				case ThModeXM: tmode = TMPL_add_var( NULL, "THMODE", "Mage single-player", NULL); break;
				case ThModeMP: tmode = TMPL_add_var( NULL, "THMODE", (secondDM ? "Multiplayer (cont.)" : "Multiplayer"), NULL); break;
				case ThModeCP: tmode = TMPL_add_var( NULL, "THMODE", "Cooperative", NULL); break;
				case ThModeDM: tmode = TMPL_add_var( NULL, "THMODE", (secondDM ? "Deathmatch (cont.)" : "Deathmatch"), NULL); break;
			}
			/* control mode columns layout */
			if (modecount % 3 == 0)
			{
				tmode = TMPL_add_var( tmode, "THFIRST", "first", NULL);
				if (modecount > 0)
					tmode = TMPL_add_var( tmode, "THROW", "new", NULL);
			}
			modecount++;
			tmode = TMPL_add_loop( tmode, "THCLASSES", tlclasses);
			tlmodes = TMPL_add_varlist( tlmodes, tmode);
		}

		/* check whether splitting MP-only/DM table & repeat for 2nd column */
		if ((m == ThModeDM || (m == ThModeMP && WXcludeSP))
		    && WSplitDM > 0 && !secondDM)
		{
			secondDM = TRUE;
			m--; // fool the modes loop through MP-only/DM again
		}
	}
	return TMPL_add_loop( tpl, "THMODES", tlmodes);
}

/*
	add integer as string to template variables
*/
TMPL_varlist *tmpl_add_int( TMPL_varlist *tpl, char *name, BCLNG value)
{
	char strbuf[12];

	snprintf( strbuf, sizeof(strbuf), "%u", value);

	return TMPL_add_var( tpl, name, strbuf, NULL);
}

/*
	add skill anchor as string to template variables
*/
TMPL_varlist *tmpl_add_skill( TMPL_varlist *tpl)
{
	char *skill;

	switch (GameVersion & 0xF0)
	{
		case 0x00: skill = "Doom and Doom II skill levels"; break;
		case 0x10: skill = "Heretic skill levels"; break;
		case 0x20: skill = "Strife skill levels"; break;
		case 0x40: skill = "Hexen skill levels"; break;
	}
	return TMPL_add_var( tpl, "THSKILL", skill, NULL);
}


#define PlayerEnd 8

/*
	perform aggregation of special things
*/
void AggregateSpecials( void)
{
	int i, j, m, p;
	int players[PlayerEnd];
	Bool found;

	/* init player things to aggregate */
	players[0] = THING_PLAYER1;
	players[1] = THING_PLAYER2;
	players[2] = THING_PLAYER3;
	players[3] = THING_PLAYER4;
	switch (GameVersion & 0xF0)
	{
		case 0x00:
			players[4] = THING_PLAYER5D;
			players[5] = THING_PLAYER6D;
			players[6] = THING_PLAYER7D;
			players[7] = THING_PLAYER8D;
			break;
		case 0x10:
			players[4] = THING_PLAYER5H;
			players[5] = THING_PLAYER6H;
			players[6] = THING_PLAYER7H;
			players[7] = THING_PLAYER8H;
			break;
		case 0x20:
			players[4] = THING_PLAYER5S;
			players[5] = THING_PLAYER6S;
			players[6] = THING_PLAYER7S;
			players[7] = THING_PLAYER8S;
			break;
		case 0x40:
			players[4] = THING_PLAYER5X;
			players[5] = THING_PLAYER6X;
			players[6] = THING_PLAYER7X;
			players[7] = THING_PLAYER8X;
			break;
	}

	/* find coop starts item */
	j = -1;
	for (i = 0; i < NumThItems; i++)
		if (ThItems[i]->type == THING_COOPSTARTS)
		{
			j = i;
			break;
		}
	/* count distinct player starts as coop */
	if (j != -1)
	{
		for (i = 0; i < NumThItems; i++)
			for (p = 0; p < PlayerEnd; p++)
				if (players[p] == ThItems[i]->type)
					for (m = 0; m < ThMode_End; m++)
					{
						ThItems[j]->easy[m] += (ThItems[i]->easy[m] > 0 ? 1 : 0);
						ThItems[j]->medm[m] += (ThItems[i]->medm[m] > 0 ? 1 : 0);
						ThItems[j]->hard[m] += (ThItems[i]->hard[m] > 0 ? 1 : 0);
					}
		/* clear coop starts if just one */
		for (m = 0; m < ThMode_End; m++)
			if (ThItems[j]->easy[m] == ThItems[j]->medm[m] &&
			    ThItems[j]->easy[m] == ThItems[j]->hard[m] &&
			    ThItems[j]->easy[m] == 1)
				ThItems[j]->easy[m] = ThItems[j]->medm[m] = ThItems[j]->hard[m] = 0;
	}

	/* find voodoo dolls item */
	j = -1;
	for (i = 0; i < NumThItems; i++)
		if (ThItems[i]->type == THING_VOODOODOLLS)
		{
			j = i;
			break;
		}
	/* count duplicate+ player starts as dolls */
	if (j != -1)
	{
		for (i = 0; i < NumThItems; i++)
			for (p = 0; p < PlayerEnd; p++)
				if (players[p] == ThItems[i]->type)
					for (m = 0; m < ThMode_End; m++)
					{
						ThItems[j]->easy[m] += (ThItems[i]->easy[m] > 0 ? ThItems[i]->easy[m]-1 : 0);
						ThItems[j]->medm[m] += (ThItems[i]->medm[m] > 0 ? ThItems[i]->medm[m]-1 : 0);
						ThItems[j]->hard[m] += (ThItems[i]->hard[m] > 0 ? ThItems[i]->hard[m]-1 : 0);
					}
	}

	/* clear player things */
	for (i = 0; i < NumThItems; i++)
		for (p = 0; p < PlayerEnd; p++)
			if (players[p] == ThItems[i]->type)
				for (m = 0; m < ThMode_End; m++)
					ThItems[i]->easy[m] = ThItems[i]->medm[m] = ThItems[i]->hard[m] = 0;
}

/*
	check whether to list this thing in this mode
*/
Bool ThingInMode( UBCINT type, int mode)
{
	/* DM & Team starts only in MP & DM modes */
	if (type == THING_DEATHMATCH ||
	    type == THING_TEAMBLUE  || type == THING_TEAMRED ||
	    type == THING_TEAMGREEN || type == THING_TEAMGOLD)
	{
		if (mode == ThModeMP || mode == ThModeDM)
			return TRUE;
		else
			return FALSE;
	}

	/* Coop starts only in MP & Coop modes, unless excluding SP/Coop */
	if (type == THING_COOPSTARTS)
	{
		if (!WXcludeSP && (mode == ThModeMP || mode == ThModeCP))
			return TRUE;
		else
			return FALSE;
	}

	return TRUE;
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
