/*
	DooM MaP StaTistics, by Frans P. de Vries.

Derived from:

	DooM PostScript Maps Utility, by Frans P. de Vries.

And thus from:

	Doom Editor Utility, by Brendon Wyber and Raphaël Quinet.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	LEVELS.C - Level loading and saving routines.
*/

#include "dmmpst.h"
#include "levels.h"


/*
	the global variables
*/
MDirPtr  Level = NULL;     /* master dictionary entry for the level */
char    *LevelName = NULL; /* official name for the level */

UBCINT NumThings = 0;      /* number of things */
TPtr      Things;          /* things data */
UBCINT NumLineDefs = 0;    /* number of line defs */
LDPtr     LineDefs;        /* line defs data */
UBCINT NumSideDefs = 0;    /* number of side defs */
SDPtr     SideDefs;        /* side defs data */
BCLNG  TotVertexes = 0;    /* total number of vertexes */
BCLNG  NumVertexes = 0;    /* number of used vertexes */
VPtr      Vertexes;        /* vertex data */
UBCINT NumSegs = 0;        /* number of segments */
SEPtr     Segs = NULL,     /* list of segments */
       LastSeg = NULL;     /* last segment in the list */
UBCINT NumSSectors = 0;    /* number of subsectors */
SSPtr     SSectors = NULL, /* list of subsectors */
       LastSSector = NULL; /* last subsector in the list */
UBCINT NumSectors = 0;     /* number of sectors */
SPtr      Sectors;         /* sectors data */

BCINT MapMaxX = -32767;    /* maximum X value of map */
BCINT MapMaxY = -32767;    /* maximum Y value of map */
BCINT MapMinX = 32767;     /* minimum X value of map */
BCINT MapMinY = 32767;     /* minimum Y value of map */


/* official names for all Ultimate DOOM levels */
char *LevelNames1[4][9] = {
	{ "Hangar", "Nuclear Plant", "Toxin Refinery", "Command Control", "Phobos Lab",
	  "Central Processing", "Computer Station", "Phobos Anomaly", "Military Base" },
	{ "Deimos Anomaly", "Containment Area", "Refinery", "Deimos Lab", "Command Center",
	  "Halls of the Damned", "Spawning Vats", "Tower of Babel", "Fortress of Mystery" },
	{ "Hell Keep", "Slough of Despair", "Pandemonium", "House of Pain",
	  "Unholy Cathedral", "Mt. Erebus", "Limbo", "Dis", "Warrens" },
	{ "Hell Beneath", "Perfect Hatred", "Sever the Wicked", "Unruly Evil", "They Will Repent",
	  "Against Thee Wickedly", "And Hell Followed", "Unto the Cruel", "Fear" }
};

/* official names for all DOOM II levels */
char *LevelNames2[32] = {
	"Entryway", "Underhalls", "The Gantlet", "The Focus", "The Waste Tunnels", "The Crusher",
	"Dead Simple", "Tricks and Traps", "The Pit", "Refueling Base", "'O' of Destruction!",
	"The Factory", "Downtown", "The Inmost Dens", "Industrial Zone", "Suburbs", "Tenements",
	"The Courtyard", "The Citadel", "Gotcha!", "Nirvana", "The Catacombs", "Barrels o' Fun",
	"The Chasm", "Bloodfalls", "The Abandoned Mines", "Monster Condo", "The Spirit World",
	"The Living End", "Icon of Sin", "Wolfenstein", "Grosse"
};

/* official names for all TNT: Evilution levels */
char *LevelNamesT[32] = {
	"System Control", "Human BBQ", "Power Control", "Wormhole", "Hanger", "Open Season",
	"Prison", "Metal", "Stronghold", "Redemption", "Storage Facility", "Crater",
	"Nukage Processing", "Steel Works", "Dead Zone", "Deepest Reaches", "Processing ",
	"Mill", "Shipping/Respawning", "Central Processing", "Administration Center",
	"Habitat", "Lunar Mining Project", "Quarry", "Baron's Den", "Ballistyx",
	"Mount Pain", "Heck", "River Styx", "Last Call", "Pharaoh", "Caribbean"
};

/* official names for all Plutonia Experiment levels */
char *LevelNamesP[32] = {
	"Congo", "Well of Souls", "Aztec", "Caged", "Ghost Town", "Baron's Lair",
	"Caughtyard", "Realm", "Abattoire", "Onslaught", "Hunted", "Speed", "The Crypt",
	"Genesis", "The Twilight", "The Omen", "Compound", "Neurosphere", "NME",
	"The Death Domain", "Slayer", "Impossible Mission", "Tombstone", "The Final Frontier",
	"The Temple of Darkness", "Bunker", "Anti-Christ", "The Sewers", "Odyssey of Noises",
	"The Gateway of Hell", "Cyberden", "Go 2 It"
};

/* official names for all Heretic: Shadow of the Serpent Riders levels */
char *LevelNamesH[6][9] = {
	{ "The Docks", "The Dungeons", "The Gatehouse", "The Guard Tower", "The Citadel",
	  "The Cathedral", "The Crypts", "Hell's Maw", "The Graveyard" },
	{ "The Crater", "The Lava Pits", "The River of Fire", "The Ice Grotto", "The Catacombs",
	  "The Labyrinth", "The Great Hall", "The Portals of Chaos", "The Glacier" },
	{ "The Storehouse", "The Cesspool", "The Confluence", "The Azure Fortress",
	  "The Ophidian Lair", "The Halls of Fear", "The Chasm", "D'Sparil's Keep", "The Aquifer" },
	{ "Catafalque", "Blockhouse", "Ambulatory", "Sepulcher", "Great Stair",
	  "Halls of the Apostate", "Ramparts of Perdition", "Shattered Bridge", "Mausoleum" },
	{ "Ochre Cliffs", "Rapids", "Quay", "Courtyard", "Hydratyr",
	  "Colonnade", "Foetid Manse", "Field of Judgement", "Skein of D'Sparil" },
	{ "Raven's Lair", "The Water Shrine", "American's Legacy", "", "", "", "", "", "" }
};

/* official names for all Hexen levels */
char *LevelNamesX[40] = {
	"Hub 1: Winnowing Hall", "Hub 1: Seven Portals", "Hub 1: Guardian of Ice",
	"Hub 1: Guardian of Fire", "Hub 1: Guardian of Steel", "Hub 1: Bright Crucible",
	"" /* 7 */, "Hub 2: Darkmere", "Hub 2: Caves of Circe", "Hub 2: Wastelands",
	"Hub 2: Sacred Grove", "Hub 2: Hypostyle", "Hub 2: Shadow Wood", "" /* 14 */,
	"" /* 15 */, "" /* 16 */, "" /* 17 */, "" /* 18 */, "" /* 19 */, "" /* 20 */,
	"Hub 4: Forsaken Outpost", "Hub 4: Castle of Grief", "Hub 4: Gibbet",
	"Hub 4: Effluvium", "Hub 4: Dungeons", "Hub 4: Desolate Garden",
	"Hub 3: Heresiarch's Seminary", "Hub 3: Dragon Chapel", "" /* 29 */,
	"Hub 3: Griffin Chapel", "Hub 3: Deathwind Chapel", "Hub 3: Orchard of Lamentations",
	"Hub 3: Silent Refectory", "Hub 3: Wolf Chapel", "Hub 5: Necropolis",
	"Hub 5: Zedek's Tomb", "Hub 5: Menelkir's Tomb", "Hub 5: Traductus' Tomb",
	"Hub 5: Vivarium", "Hub 5: Dark Crucible"
};

/* official names for all Hexen: Deathkings of the Dark Citadel levels */
char *LevelNamesD[30] = {
	"" /* 31 */, "" /* 32 */, "Hub 4: Transit", "Hub 4: Over N Under", "Hub 4: Deathfog",
	"Hub 4: Castle of Pain", "Hub 4: Sewer Pit", "Hub 4: The Rose", "" /* 39 */,  "" /* 40 */,
	"Hub 1: Ruined Village", "Hub 1: Blight", "Hub 1: Sump", "Hub 1: Catacomb",
	"Hub 1: Badlands", "Hub 1: Brackenwood", "Hub 1: Pyre", "Hub 2: Constable's Gate",
	"Hub 2: Treasury", "Hub 2: Market Place", "Hub 2: Locus Requiescat", "Hub 2: Ordeal",
	"Hub 2: Armory", "Hub 3: Nave", "Hub 3: Chantry", "Hub 3: Abattoir", "Hub 3: Dark Watch", "Hub 3: Cloaca", "Hub 3: Ice Hold", "Hub 3: Dark Citadel"
};

/* official names for all Strife levels */
char *LevelNamesS[38] = {
	"Sanctuary", "Town", "Front Base", "Power Station", "Prison", "Sewers", "Castle",
	"Audience Chamber", "Castle: Programmer's Keep", "New Front Base", "Borderlands",
	"The Temple of the Oracle", "Catacombs", "Mines", "Fortress: Administration",
	"Fortress: Bishop's Tower", "Fortress: The Bailey", "Fortress: Stores",
	"Fortress: Security Complex", "Factory: Receiving", "Factory: Manufacturing",
	"Factory: Forge", "Order Commons", "Factory: Conversion Chapel", "Catacombs: Ruined Temple",
	"Proving Grounds", "The Lab", "Alien Ship", "Entity's Lair", "Abandoned Front Base",
	"Training Facility", "Sanctuary", "Town", "Movement Base",
	/* Veteran Edition */
	"Factory: Production", "Castle Clash", "Killing Grounds", "Ordered Chaos"
};


/*
	check the input level parameters
*/
void CheckLevelParams( BCINT episode, BCINT mission)
{
	BCINT MaxEpisode, MinMission, MaxMission;

	if (GameVersion == 0x02 || GameVersion >= 0x20)
	{
		if (episode != 0)
			ProgError( "invalid game episode number (%d)", episode);
		switch (GameVersion)
		{
			case 0x02: MinMission =  1; MaxMission = 59; break; // Doom II + megawads
			case 0x20: MinMission = 32; MaxMission = 34; break; // Strife demo
			case 0x22: MinMission =  1; MaxMission = 38; break; // Strife full
			case 0x40: MinMission =  1; MaxMission =  4; break; // Hexen demo
			case 0x42: MinMission =  1; MaxMission = 40; break; // Hexen full
			case 0x48: MinMission = 33; MaxMission = 60; break; // Hexen DDC
			default: ProgError( "unsupported game version (%02x)", GameVersion);
		}
		if (mission < MinMission || mission > MaxMission)
			ProgError( "invalid or missing game mission number (%d)", mission);
	}
	else
	{
		MaxEpisode = 1;
		MinMission = 1;
		MaxMission = 9;
		switch (GameVersion)
		{
			case 0x00: break; // Doom shareware
			case 0x01: MaxEpisode = 3; break; // Doom reg
			case 0x04: MaxEpisode = 6; MinMission = 0; break; // Doom ult + megawads
			case 0x10: break; // Heretic shareware
			case 0x11: MaxEpisode = 4; /* check for & allow Heretic E4M1 */
			           if (episode == MaxEpisode) MaxMission = 1; break; // Heretic reg
			case 0x14: MaxEpisode = 6; /* check for & allow Heretic:SotSR E6M1-3 */
			           if (episode == MaxEpisode) MaxMission = 3; break; // Heretic SSR
			default: ProgError( "unsupported game version (%02x)", GameVersion);
		}
		if (episode < 1 || episode > MaxEpisode)
			ProgError( "invalid or missing game episode number (%d)", episode);
		if (mission < MinMission || mission > MaxMission)
			ProgError( "invalid or missing game mission number (%d)", mission);
	}
}


/*
	read in the level data
*/
BCINT ReadLevelData( BCINT episode, BCINT mission)
{
	MDirPtr dir;
	char name[9];
	BCLNG n, m;
	BCINT val, lumpver = GameVersion;
	Bool *VertexUsed;

	/* find the various level information from the master directory */
	if (GameVersion == 0x02 || GameVersion >= 0x20)
		sprintf( name, "MAP%02d", mission);
	else
		sprintf( name, "E%dM%d", episode, mission);
	Level = FindMasterDir( MasterDir, name);
	if (!Level)
		ProgError( "level data not found for %d %d", episode, mission);

	/* check for Hexen-format PWAD loaded with Doom-series IWAD */
	if (GameVersion < 0x10 && FindLevelDir( Level, "BEHAVIOR") != NULL)
		lumpver += 0x40;

	if (GameVersion == 0x02 || GameVersion >= 0x20)
	{
		if ((GameVersion & 0xF0) == 0x40)
			if (GameVersion == 0x48)
				LevelName = LevelNamesD[mission-31]; /* Hexen: DotDC */
			else /* GameVersion == 0x40 or 0x42 */
				LevelName = LevelNamesX[mission-1]; /* Hexen */
		else if ((GameVersion & 0xF0) == 0x20)
			LevelName = LevelNamesS[mission-1]; /* Strife */
		else /* GameVersion == 0x02 */
			if (mission <= 32)
				LevelName = LevelNames2[mission-1]; /* Doom II */
			else
				LevelName = ""; /* megawads */
	}
	else
	{
		if ((GameVersion & 0xF0) == 0x10)
		{
			if (GameVersion == 0x11 && episode == 4 && mission == 1) /* Heretic E4M1 */
				LevelName = LevelNamesH[5][2]; /* == Heretic: SotSR E6M3 */
			else /* GameVersion == 0x10, 0x11, or 0x14 */
				LevelName = LevelNamesH[episode-1][mission-1];
		}
		else /* GameVersion == 0x00, 0x01, or 0x04 */
			LevelName = LevelNames1[episode-1][mission-1];
	}

	/* read in the Things data */
	dir = FindLevelDir( Level, "THINGS");
	if (dir)
	{
		if ((lumpver & 0xF0) == 0x40) // Hexen format
			NumThings = (UBCINT) (dir->dir.size / sizeof( struct ThingH));
		else // Doom format
			NumThings = (UBCINT) (dir->dir.size / sizeof( struct Thing));
	}
	else
		NumThings = 0;
	if (NumThings > 0)
	{
		Things = (TPtr) GetFarMemory( NumThings * sizeof( struct Thing));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumThings; n++)
		{
			if ((lumpver & 0xF0) == 0x40) // Hexen format, skip what's not needed
			{
				BasicWadRead( dir->wadfile, &val, 2); // tid
			}
			BasicWadRead( dir->wadfile, &(Things[n].xpos), 2);
			swapint( &(Things[n].xpos));
			BasicWadRead( dir->wadfile, &(Things[n].ypos), 2);
			swapint( &(Things[n].ypos));
			if ((lumpver & 0xF0) == 0x40) // Hexen format
			{
				BasicWadRead( dir->wadfile, &val, 2); // zpos
			}
			BasicWadRead( dir->wadfile, &(Things[n].angle), 2);
			swapint( &(Things[n].angle));
			BasicWadRead( dir->wadfile, &(Things[n].type), 2);
			swapint( &(Things[n].type));
			BasicWadRead( dir->wadfile, &(Things[n].when), 2);
			swapint( &(Things[n].when));
			if ((lumpver & 0xF0) == 0x40) // Hexen format
			{
				BasicWadRead( dir->wadfile, &val, 2); // special, arg1
				BasicWadRead( dir->wadfile, &val, 2); // arg2, arg3
				BasicWadRead( dir->wadfile, &val, 2); // arg4, arg5
			}
		}
	}

	/* get the number of Vertices */
	dir = FindLevelDir( Level, "VERTEXES");
	if (dir)
		TotVertexes = (BCLNG) (dir->dir.size / sizeof( struct Vertex));
	else
		TotVertexes = 0;
	if (TotVertexes > 0)
	{
		VertexUsed = (Bool *) GetMemory( TotVertexes * sizeof( Bool));
		for (n = 0; n < TotVertexes; n++)
			VertexUsed[n] = FALSE;
	}

	/* read in the LineDef information */
	dir = FindLevelDir( Level, "LINEDEFS");
	if (dir)
	{
		if ((lumpver & 0xF0) == 0x40) // Hexen format
			NumLineDefs = (UBCINT) (dir->dir.size / sizeof( struct LineDefH));
		else
			NumLineDefs = (UBCINT) (dir->dir.size / sizeof( struct LineDef));
	}
	else
		NumLineDefs = 0;
	if (NumLineDefs > 0)
	{
		LineDefs = (LDPtr) GetFarMemory( NumLineDefs * sizeof( struct LineDef));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumLineDefs; n++)
		{
			BasicWadRead( dir->wadfile, &(LineDefs[n].start), 2);
			swapint( &(LineDefs[n].start));
			VertexUsed[LineDefs[n].start] = TRUE;
			BasicWadRead( dir->wadfile, &(LineDefs[n].end), 2);
			swapint( &(LineDefs[n].end));
			VertexUsed[LineDefs[n].end] = TRUE;
			BasicWadRead( dir->wadfile, &(LineDefs[n].flags), 2);
			swapint( &(LineDefs[n].flags));
			if ((lumpver & 0xF0) == 0x40) // Hexen format, skip what's not needed
			{
				BasicWadRead( dir->wadfile, &val, 2); // special, arg1
				BasicWadRead( dir->wadfile, &val, 2); // arg2, arg3
				BasicWadRead( dir->wadfile, &val, 2); // arg4, arg5
				LineDefs[n].type = 0;
				LineDefs[n].tag = 0;
			}
			else // Doom format
			{
				BasicWadRead( dir->wadfile, &(LineDefs[n].type), 2);
				swapint( &(LineDefs[n].type));
				BasicWadRead( dir->wadfile, &(LineDefs[n].tag), 2);
				swapint( &(LineDefs[n].tag));
			}
			BasicWadRead( dir->wadfile, &(LineDefs[n].sidedef1), 2);
			swapint( &(LineDefs[n].sidedef1));
			BasicWadRead( dir->wadfile, &(LineDefs[n].sidedef2), 2);
			swapint( &(LineDefs[n].sidedef2));
		}
	}

	/* read in the SideDef information */
	dir = FindLevelDir( Level, "SIDEDEFS");
	if (dir)
		NumSideDefs = (UBCINT) (dir->dir.size / sizeof( struct SideDef));
	else
		NumSideDefs = 0;
	if (NumSideDefs > 0)
	{
		SideDefs = (SDPtr) GetFarMemory( NumSideDefs * sizeof( struct SideDef));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumSideDefs; n++)
		{
			BasicWadRead( dir->wadfile, &(SideDefs[n].xoff), 2);
			swapint( &(SideDefs[n].xoff));
			BasicWadRead( dir->wadfile, &(SideDefs[n].yoff), 2);
			swapint( &(SideDefs[n].yoff));
			BasicWadRead( dir->wadfile, &(SideDefs[n].tex1), 8);
			BasicWadRead( dir->wadfile, &(SideDefs[n].tex2), 8);
			BasicWadRead( dir->wadfile, &(SideDefs[n].tex3), 8);
			BasicWadRead( dir->wadfile, &(SideDefs[n].sector), 2);
			swapint( &(SideDefs[n].sector));
		}
	}

	/* read in the Vertices which are all the corners of the level, but ignore the */
	/* Vertices not used in any LineDef (they usually are at the end of the list). */
	NumVertexes = 0;
	for (n = 0; n < TotVertexes; n++)
		if (VertexUsed[n])
			NumVertexes++;
	if (NumVertexes > 0)
	{
		Vertexes = (VPtr) GetFarMemory( NumVertexes * sizeof( struct Vertex));
		dir = FindLevelDir( Level, "VERTEXES");
		BasicWadSeek( dir->wadfile, dir->dir.start);
		MapMaxX = -32767;
		MapMaxY = -32767;
		MapMinX = 32767;
		MapMinY = 32767;
		m = 0;
		for (n = 0; n < TotVertexes; n++)
		{
			BasicWadRead( dir->wadfile, &val, 2);
			swapint( &val);
			if (VertexUsed[n])
			{
				if (val < MapMinX)
					MapMinX = val;
				if (val > MapMaxX)
					MapMaxX = val;
				Vertexes[m].x = val;
			}
			BasicWadRead( dir->wadfile, &val, 2);
			swapint( &val);
			if (VertexUsed[n])
			{
				if (val < MapMinY)
					MapMinY = val;
				if (val > MapMaxY)
					MapMaxY = val;
				Vertexes[m].y = val;
				m++;
			}
		}
		if (m != NumVertexes)
			ProgError("inconsistency in the Vertexes data");
	}

	if (TotVertexes > 0)
	{
		/* update the Vertex numbers in the LineDefs (not really necessary, but...) */
		m = 0;
		for (n = 0; n < TotVertexes; n++)
			if (VertexUsed[n])
				VertexUsed[n] = m++;
		for (n = 0; n < NumLineDefs; n++)
		{
			LineDefs[n].start = VertexUsed[LineDefs[n].start];
			LineDefs[n].end = VertexUsed[LineDefs[n].end];
		}
		FreeMemory( VertexUsed);
		VertexUsed = NULL;
	}

	/* ignore the Segs, SSectors and Nodes */

	/* read in the Sectors information */
	dir = FindLevelDir( Level, "SECTORS");
	if (dir)
		NumSectors = (UBCINT) (dir->dir.size / sizeof( struct Sector));
	else
		NumSectors = 0;
	if (NumSectors > 0)
	{
		Sectors = (SPtr) GetFarMemory( NumSectors * sizeof( struct Sector));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumSectors; n++)
		{
			BasicWadRead( dir->wadfile, &(Sectors[n].floorh), 2);
			swapint( &(Sectors[n].floorh));
			BasicWadRead( dir->wadfile, &(Sectors[n].ceilh), 2);
			swapint( &(Sectors[n].ceilh));
			BasicWadRead( dir->wadfile, &(Sectors[n].floort), 8);
			BasicWadRead( dir->wadfile, &(Sectors[n].ceilt), 8);
			BasicWadRead( dir->wadfile, &(Sectors[n].light), 2);
			swapint( &(Sectors[n].light));
			BasicWadRead( dir->wadfile, &(Sectors[n].special), 2);
			swapint( &(Sectors[n].special));
			BasicWadRead( dir->wadfile, &(Sectors[n].tag), 2);
			swapint( &(Sectors[n].tag));
		}
	}

	/* ignore the Reject and BlockMap, and Hexen/other lumps */

	return lumpver;
}


/*
	forget the level data
*/
void ForgetLevelData( void)
{
	/* forget the Things */
	if (Things)
		FreeFarMemory( Things);
	Things = NULL;
	NumThings = 0;

	/* forget the Vertices */
	if (Vertexes)
		FreeFarMemory( Vertexes);
	Vertexes = NULL;
	NumVertexes = 0;

	/* forget the LineDefs */
	if (LineDefs)
		FreeFarMemory( LineDefs);
	LineDefs = NULL;
	NumLineDefs = 0;

	/* forget the SideDefs */
	if (SideDefs)
		FreeFarMemory( SideDefs);
	SideDefs = NULL;
	NumSideDefs = 0;

	/* forget the Sectors */
	if (Sectors)
		FreeFarMemory( Sectors);
	Sectors = NULL;
	NumSectors = 0;

	/* forget the level pointers */
	Level = NULL;
	LevelName = NULL;
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
