/*
	DooM MaP StaTistics, by Frans P. de Vries.

Derived from:

	DooM PostScript Maps Utility, by Frans P. de Vries.

And thus from:

	Doom Editor Utility, by Brendon Wyber and Raphaël Quinet.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	DMMPST.C - Main program routines
*/

#include <signal.h>
#include "dmmpst.h"
#include "stats.h"
#include "appear.h"


/*
	the global variables
*/
BCINT GameVersion = 0x00; /* which game and version? */
                          /* if you change this, bad things will happen to you... */
                 /* 0x00 = Shareware / Demo
                    0x01 = Registered
                    0x02 = Commercial (II/Final, Strife, Hexen)
                    0x04 = Ultimate / Shadow of the Serpent Riders
                    0x08 = Deathkings of the Dark Citadel
                   +0x00 = Doom
                   +0x10 = Heretic
                   +0x20 = Strife
                   +0x40 = Hexen
                  */
char *MainWad     = "DOOM.WAD"; /* name of the main WAD file */
char **PatchWads  = NULL;  /* list of patch WAD files */
char *UserLvlNm   = NULL;  /* user defined level name */
char *UserGamNm   = NULL;  /* user defined game/megawad name */
char *CustomMap   = NULL;  /* path to custom Things translations */
char *WkTpPath    = NULL;  /* path to input DoomWiki templates */
FILE *WkFile      = NULL;  /* the output DoomWiki file */
Bool Verbose      = FALSE; /* verbose logging & statistics? */
Bool WSkeleton    = FALSE; /* skeleton wiki? */
Bool WSecrets     = FALSE; /* secrets wiki? */
Bool WStatistics  = FALSE; /* statistics wiki? */
Bool WMapdata     = FALSE; /* map data wiki? */
Bool WThings      = FALSE; /* things wiki? */
Bool WBoomMP      = FALSE; /* Boom multiplayer in things wiki? */
Bool WXcludeSP    = FALSE; /* exclude SP/Coop, only MP/DM things? */
BCLNG WSplitDM    = 0;     /* split DM table by classes */
BCLNG AppearType  = 0;     /* tally appearance by thing type */


/* local variables */
char *WkFlName   = NULL; /* name of the output DoomWiki file */
BCINT Episode    = 0;    /* the episode number */
BCINT Mission    = 0;    /* the mission number */

OptDesc options[] =    /* description of the command line options */
{
/* short/long names  type  message if true/changed  message if false  where to store value */
	{ "W",  "MAIN",      OPT_STRING,    "Main WAD file",           NULL, &MainWad     },
	{ "PW", "PWAD",      OPT_STRINGACC, "Patch WAD file",          NULL, &PatchWads   },
	{ "",   "FILE",      OPT_STRINGLIST,"Patch WAD file",          NULL, &PatchWads   },
	{ "G",  "GAME",      OPT_STRING,    "User game/megawad name",  NULL, &UserGamNm   },
	{ "N",  "NAME",      OPT_STRING,    "User level name",         NULL, &UserLvlNm   },
	{ "E",  "EPISODE",   OPT_INTEGER,   "Episode number",          NULL, &Episode     },
	{ "M",  "MISSION",   OPT_INTEGER,   "Mission number",          NULL, &Mission     },
	{ "C",  "CUSTOMMAP", OPT_STRING,    "Custom Things mapping",   NULL, &CustomMap   },
	{ "I",  "WIKITEMPL", OPT_STRING,    "DoomWiki templates dir",  NULL, &WkTpPath    },
	{ "O",  "WIKIFILE",  OPT_STRING,    "DoomWiki output file",    NULL, &WkFlName    },
	{ "V",  "VERBOSE",   OPT_BOOLEAN,   "Verbose log & statistics",NULL, &Verbose     },
	{ "K",  "SKELETON",  OPT_BOOLEAN,   "Skeleton wiki output",    NULL, &WSkeleton   },
	{ "R",  "SECRETS",   OPT_BOOLEAN,   "Secrets wiki output",     NULL, &WSecrets    },
	{ "S",  "STATISTICS",OPT_BOOLEAN,   "Statistics wiki output",  NULL, &WStatistics },
	{ "D",  "MAPDATA",   OPT_BOOLEAN,   "Map data wiki output",    NULL, &WMapdata    },
	{ "T",  "THINGS",    OPT_BOOLEAN,   "Things wiki output",      NULL, &WThings     },
	{ "B",  "BOOMMP",    OPT_BOOLEAN,   "Boom multiplayer things", NULL, &WBoomMP     },
	{ "X",  "EXCLUDESP", OPT_BOOLEAN,   "Exclude SP/Coop things",  NULL, &WXcludeSP   },
	{ "Z",  "SPLITDM",   OPT_LONG,      "Split DM table by class", NULL, &WSplitDM    },
	{ "A",  "APPEARANCE",OPT_INTEGER,   "Appearance of thing type",NULL, &AppearType  },
	{ NULL, NULL,        OPT_END,       NULL,                      NULL, NULL         }
};


/* local function definitions */
void ParseCommandLineOptions( int, char *[]);
void Usage( FILE *);
void Credits( FILE *);
void InitSignal( void);
void CatchSignal( int);
void MapStats( void);
void ThingStats( void);


/*
	the main program
*/
int main( int argc, char *argv[])
{
	BCINT i;

	InitSignal();
	Credits( stdout);
	argv++;
	argc--;
	/* read command line options */
	ParseCommandLineOptions( argc, argv);

	/* check requested stats */
	if (AppearType != 0)
		ThingStats();
	else
		MapStats();

	exit( 0);
}


/*
	append a string to a null-terminated string list
*/
void AppendItemToList( char ***list, char *item)
{
	BCINT i = 0;

	if (*list)
	{
		/* count the number of elements in the list (last = null) */
		while ((*list)[i])
			i++;
		/* expand the list */
		*list = (char **) ResizeMemory( *list, (i + 2) * sizeof( char **));
	}
	else
		/* create a new list */
		*list = (char **) GetMemory( 2 * sizeof( char **));

	/* append the new element */
	(*list)[i] = item;
	(*list)[i + 1] = NULL;
}


/*
	handle command line options
*/
void ParseCommandLineOptions( int argc, char *argv[])
{
	BCINT optnum;

	while (argc > 0)
	{
		if (argv[0][0] != '-' && argv[0][0] != '+')
			ProgError( "options must start with '-' or '+'");
		strupr( argv[0]);
		if (strcmp( argv[0], "-?") == 0 ||
		    strcmp( argv[0], "-H") == 0 ||
		    strcmp( argv[0], "-HELP") == 0)
		{
			Usage( stdout);
			exit( 0);
		}
		for (optnum = 0; options[optnum].opt_type != OPT_END; optnum++)
		{
			if (strcmp( &(argv[0][1]), options[optnum].short_name) == 0 ||
			    strcmp( &(argv[0][1]), options[optnum].long_name) == 0)
			{
				switch (options[optnum].opt_type)
				{
				case OPT_BOOLEAN:
					if (argv[0][0] == '+') /* '+/-' reversed for print flags */
					{
						*((Bool *) (options[optnum].data_ptr)) = TRUE;
						if (options[optnum].msg_if_true)
							printf( "%s.\n", options[optnum].msg_if_true);
					}
					else
					{
						*((Bool *) (options[optnum].data_ptr)) = FALSE;
						if (options[optnum].msg_if_false)
							printf( "%s.\n", options[optnum].msg_if_false);
					}
					break;
				case OPT_INTEGER:
					if (argc <= 1)
						ProgError( "missing argument after \"%s\"", argv[0]);
					argv++;
					argc--;
					*((BCINT *) (options[optnum].data_ptr)) = atoi( argv[0]);
					if (options[optnum].msg_if_true)
						printf( "%s: %d.\n", options[optnum].msg_if_true, atoi( argv[0]));
					break;
				case OPT_LONG:
					if (argc <= 1)
						ProgError( "missing argument after \"%s\"", argv[0]);
					argv++;
					argc--;
					*((BCLNG *) (options[optnum].data_ptr)) = atoi( argv[0]);
					if (options[optnum].msg_if_true)
						printf( "%s: %d.\n", options[optnum].msg_if_true, atoi( argv[0]));
					break;
				case OPT_STRING:
					if (argc <= 1)
						ProgError( "missing argument after \"%s\"", argv[0]);
					argv++;
					argc--;
					*((char **) (options[optnum].data_ptr)) = argv[0];
					if (options[optnum].msg_if_true)
						printf( "%s: %s.\n", options[optnum].msg_if_true, argv[0]);
					break;
				case OPT_STRINGACC:
					if (argc <= 1)
						ProgError( "missing argument after \"%s\"", argv[0]);
					argv++;
					argc--;
					AppendItemToList( (char ***) options[optnum].data_ptr, argv[0]);
					if (options[optnum].msg_if_true)
						printf( "%s: %s.\n", options[optnum].msg_if_true, argv[0]);
					break;
				case OPT_STRINGLIST:
					if (argc <= 1)
						ProgError( "missing argument after \"%s\"", argv[0]);
					while (argc > 1 && argv[1][0] != '-' && argv[1][0] != '+')
					{
						argv++;
						argc--;
						AppendItemToList( (char ***) options[optnum].data_ptr, argv[0]);
						if (options[optnum].msg_if_true)
							printf( "%s: %s.\n", options[optnum].msg_if_true, argv[0]);
					}
					break;
				default:
					ProgError( "unknown option type (BUG!)");
				}
				break;
			}
		}
		if (options[optnum].opt_type == OPT_END)
			ProgError( "invalid argument: \"%s\"", argv[0]);
		argv++;
		argc--;
	}
}


/*
	output the program usage to the specified file
*/
void Usage( FILE *where)
{
	fprintf( where, "Usage:\n");
	fprintf( where, "dmmpst -w <main_wad_file> [-pw <pwad_file>] [-file <pwad_files>...]\n");
	fprintf( where, "       [+v] [+k|r|s|d|t] [+b] [+x] [-z <classes>] [-g <game>] [-n <name>]\n");
	fprintf( where, "       [-c <CustomMap_file>] [-i <DoomWiki_tpl_dir>] [-o <DoomWiki_file>]\n");
	fprintf( where, "       -e <episode> -m <mission>\n");
	fprintf( where, "dmmpst -w <main_wad_file> [-pw <iwad_file>] [-file <iwad_files>...]\n");
	fprintf( where, "       [+v] [-i <DoomWiki_tpl_dir>] -o <DoomWiki_file> -a <thing_type>\n");
	fprintf( where, "   -w    Gives the name of the main WAD file (also -main); default is DOOM.WAD\n");
	fprintf( where, "   -pw   To add one patch WAD file to be loaded; may be repeated (also -pwad)\n");
	fprintf( where, "   -file To add a list of patch (-a: main) WAD files to be loaded\n");
	fprintf( where, "   +v    Display verbose logging & statistics (also +verbose)\n");
	fprintf( where, "   +k    Generate complete skeleton to output file (also +skeleton)\n");
	fprintf( where, "   +r    Generate secrets section to output file (also +secrets)\n");
	fprintf( where, "   +s    Generate statistics section to output file (also +statistics)\n");
	fprintf( where, "   +d    Generate only map data template to output file (also +mapdata)\n");
	fprintf( where, "   +t    Generate only things template to output file (also +things)\n");
	fprintf( where, "   +b    Count Boom MP things instead of classic MP (also +boommp)\n");
	fprintf( where, "   +x    Exclude SP/Coop things, list only MP/DM (also +exludesp)\n");
	fprintf( where, "   -z    Split DM table by classes [1-9] (also +splitdm)\n");
	fprintf( where, "   -g    Defines the user game/megawad name (also -game)\n");
	fprintf( where, "   -n    Defines the user level name (also -name)\n");
	fprintf( where, "   -c    Gives the path to custom Things map file (also -custommap);\n");
	fprintf( where, "   -i    Gives the path to input DoomWiki templates (also -wikitempl);\n");
	fprintf( where, "         if templates path is missing, uses current directory\n");
	fprintf( where, "   -o    Gives the name of the output DoomWiki file (also -wikifile);\n");
	fprintf( where, "         if output file is missing, [+k|r|s|d|t] are ignored\n");
	fprintf( where, "   -e    Gives the episode number (also -episode); required, except with -a\n");
	fprintf( where, "   -m    Gives the mission number (also -mission); required, except with -a\n");
	fprintf( where, "   -a    Tally appearance statistics of a thing type for one or more IWADs\n");
	fprintf( where, "         (also -appearance); requires -o\n");
	fprintf( where, "Put a '+' instead of a '-' before boolean options to reverse their effect\n");
}


/*
	output the credits of the program to the specified file
*/
void Credits( FILE *where)
{
	fprintf( where, "DMMPST: DooM MaP StaTistics, ver %s\n", DMMPST_VERSION);
	fprintf( where, " By Frans P. de Vries <fpv@gamers.org>\n");
/*	fprintf( where, "[Derived from DEU v%s by Brendon Wyber and Raphaël Quinet]\n\n", DEU_VERSION);
	fprintf( where, "Derived from DEU: Doom Editor Utility, ver %s\n\n", DEU_VERSION);
	fprintf( where, " By Raphaël Quinet (quinet@montefiore.ulg.ac.be)\n"
	                "and Brendon J Wyber (b.wyber@csc.canterbury.ac.nz)\n"
	                " Ported to DJGPP/GO32 by Per Allansson (c91peral@und.ida.liu.se)\n"
	                "and Per Kofod (per@ollie.dnk.hp.com)\n\n"); */
}


/*
	terminate the program reporting an error
*/
void ProgError( char *errstr, ...)
{
	va_list args;

	va_start( args, errstr);
	printf( "\nProgram Error: *** ");
	vprintf( errstr, args);
	printf( " ***\n");
	va_end( args);
	/* clean up & free space */
	ForgetLevelData();
	CloseWadFiles();
	exit( 1);
}

/*
	set up segfault handler
*/
void InitSignal( void)
{
#ifdef __MINGW32__
	signal( SIGSEGV, CatchSignal);
#else
	struct sigaction sa;

	memset( &sa, 0, sizeof(sigaction));
	sigemptyset( &sa.sa_mask);
	sa.sa_handler = CatchSignal;

	sigaction( SIGSEGV, &sa, NULL);
#endif
}

/*
	catch segfault and exit
*/
void CatchSignal( int signal)
{
		ProgError( "unable to load PWAD file or level");
}


/*
	the main "program" for map statistics
*/
void MapStats( void)
{
	/* load the Wad files */
	OpenMainWad( MainWad);
	if (PatchWads)
		while (PatchWads[0])
		{
			OpenPatchWad( PatchWads[0]);
			PatchWads++;
		}
	/* sanity check */
	CloseUnusedWadFiles();

	CheckLevelParams( Episode, Mission);

	/* check for optional output file */
	if (WkFlName != NULL)
	{
		/* check for Doom II, Strife or Hexen */
		if (GameVersion == 0x02 || GameVersion >= 0x20)
			printf( "\nOutputting DoomWiki template of level MAP%02d to \"%s\".\n",
			        Mission, WkFlName);
		else
			printf( "\nOutputting DoomWiki template of level E%dM%d to \"%s\".\n",
			        Episode, Mission, WkFlName);
		if ((WkFile = fopen( WkFlName, "wt")) == NULL)
			ProgError( "error opening output file \"%s\"", WkFlName);
	}

	AnalyzeLevel( Episode, Mission);

	/* that's all, folks! */
	CloseWadFiles();
	if (WkFlName != NULL)
		fclose( WkFile);
}

/*
	the main "program" for thing statistics
*/
void ThingStats( void)
{
	/* check for required output file */
	if (WkFlName == NULL)
		ProgError( "DoomWiki output file is required");

	/* tally first Wad file */
	OpenMainWad( MainWad);
	AnalyzeAppearances();
	CloseWadFiles();

	/* tally additional Wad file(s) */
	if (PatchWads)
		while (PatchWads[0])
		{
			OpenMainWad( PatchWads[0]);
			PatchWads++;
			AnalyzeAppearances();
			CloseWadFiles();
		}

	/* write output template */
	printf( "\nOutputting DoomWiki template of thing type %d to \"%s\".\n",
	        AppearType, WkFlName);
	if ((WkFile = fopen( WkFlName, "wt")) == NULL)
		ProgError( "error opening output file \"%s\"", WkFlName);

	WriteAppearances();

	/* that's all, folks! */
	ForgetAppearances();
	fclose( WkFile);
}


/*
	convert string to uppercase
*/
char *strupr( char *str)
{
	char *s = str;

	if (str != NULL)
		while (*s)
		{
			if (islower( *s))
				*s = toupper( *s);
			s++;
		}

	return str;
}

/*
	convert string to lowercase
*/
char *strlwr( char *str)
{
	char *s = str;

	if (str != NULL)
		while (*s)
		{
			if (isupper( *s))
				*s = tolower( *s);
			s++;
		}

	return str;
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
